/*
 *     Applicazione: Verifica integrità datagrammi UDP
 *
 *           Autore: Roberto Fuligni
 *  Ultima modifica: 07/10/2022
 *
 *      Descrizione: Verifica l'integrità di un datagramma UDP attraverso
 *                   il controllo del checksum.
 */

import java.io.*;
import java.net.InetAddress;

public class VerificaIntegrita {
    final static String IpSorgente = "78.54.187.2";
    final static String IpDestinazione = "208.56.123.5";

    public static void main(String[] args) {
        if (args.length < 1)
        {
            System.err.println("Nome del file mancante.");
            System.exit(1);
        }
        verifica(args[0], IpSorgente, IpDestinazione);
    }

    private static void verifica(String nomeFile, String ipSorgente, String ipDestinazione) {
        try(var dataIn = new DataInputStream(new FileInputStream(nomeFile));
            var baos = new ByteArrayOutputStream();
            var dos = new DataOutputStream(baos)) {
            var ips = InetAddress.getByName(ipSorgente);
            var ipd = InetAddress.getByName(ipDestinazione);
            // Costruzione della struttura dati necessaria al calcolo del checksum:
            // Pseudoheader + Header + Payload

            // Fase 1: scrittura in memoria dello pseudoheader
            dos.write(ips.getAddress());
            dos.write(ipd.getAddress());
            dos.writeByte(0);                 // Zeroes
            dos.writeByte(0x11);              // Protocollo UDP
            dos.writeShort(dataIn.available());  // Lunghezza del datagramma in byte

            // Fase 2: Scrittura in memoria di header e payload: si copia in memoria
            //         il contenuto dell'intero datagramma
            dataIn.transferTo(dos);

            // Nel caso in cui la lunghezza del datagramma sia dispari, si aggiunge un
            // byte di padding.
            if (dos.size() % 2 != 0)
                dos.writeByte(0);

            dos.flush();        // Svuota i buffer interni per forzare la scrittura dei dati in RAM

            byte[] buffer = baos.toByteArray();
            if (buffer[18] == 0 && buffer[19] == 0) {
                System.err.println("Impossibile verificare l'integrità del datagramma (checksum: 0)");
                return;
            }

            int c = checksum(buffer);
            System.out.format("Datagramma %s (checksum elaborato: %04X)%n", (c == 0 ? "integro" : "corrotto"), c);

        } catch (IOException e) {
            System.out.format("Errore durante la verifica d'integrità: %s%n", e.getMessage());
        }
    }

    private static int checksum(byte[] buffer) throws IOException {
        int somma = 0;

        try(var dis = new DataInputStream(new ByteArrayInputStream(buffer))) {
            while (dis.available() > 0) {
                somma += dis.readUnsignedShort();
                if (somma > 0xFFFF)
                    somma = (somma & 0xFFFF) + (somma >> 16);
            }
            while (somma > 0xFFFF)
                somma = (somma & 0xFFFF) + (somma >> 16);

            int chk = (~somma) & 0xFFFF;
            return chk;
        }
    }
}